///////////////////////////////////////////////////////////////////////////////
//
//   CONEXANT SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   MSG1191.C - Message 1191 Processing
// 
//
//   DESCRIPTION
//
//   Functions to decode, build, and display Zodiac 1191 messages.
//
//
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h> 
#include <math.h>

#include "gentypes.h"
#include "gencons.h"
#include "ztypes.h"
#include "zcons.h"   
#include "util.h" 
#include "display.h"                

#define		Fo					10230000L
#define		XOERR_SF			0.01
#define		DOPPLER_SF			0.5
#define		DOPPLER_UNCERT_SF	0.1
#define		CODEPHASE_SF		0.001
#define     METERS_PER_CHIP		GF_SPEED_OF_LIGHT / (Fo/10)
#define		CODEPHASE_UNCERT_SF 0.001

#define		HZ_2_MPS_CF			GF_SPEED_OF_LIGHT / GF_L1 // conversion factor from Hz to meters/sec 
#define		PPM_2_MPS_CF		1e-6 * GF_SPEED_OF_LIGHT  // conversion factor from part/mil to meter/sec
#define		VALID_BUF_SIZE		9  						  // buffer to store displayed validity bits
void wait(float);  
void clear_message_line(void);   

extern FILE *MagnaMeasFilePtr;      

char   Validity_Buf[VALID_BUF_SIZE];


////////////////////////////////////////////////////////////////////////////////
//   
//   Decode a Zodiac 1191 message from the ZMsgBuf.
//
void Dec1191(tMSGBUF *ZMsgBuf, tMSG1191 *Msg)
{
	int i;
	tSatMeas *SatMeasPtr; 
	
    
    for(i=0; i<VALID_BUF_SIZE; i++) Validity_Buf[i] = ' ';
    
    (Msg->ValidityFlags & 0x0001) ? (Validity_Buf[8] = '1') : (Validity_Buf[8] = '0');
    (Msg->ValidityFlags & 0x0002) ? (Validity_Buf[7] = '1') : (Validity_Buf[7] = '0');
    (Msg->ValidityFlags & 0x0004) ? (Validity_Buf[6] = '1') : (Validity_Buf[6] = '0');
    (Msg->ValidityFlags & 0x0008) ? (Validity_Buf[5] = '1') : (Validity_Buf[5] = '0');
    (Msg->ValidityFlags & 0x0010) ? (Validity_Buf[3] = '1') : (Validity_Buf[3] = '0');
    (Msg->ValidityFlags & 0x0020) ? (Validity_Buf[2] = '1') : (Validity_Buf[2] = '0');
    (Msg->ValidityFlags & 0x0040) ? (Validity_Buf[1] = '1') : (Validity_Buf[1] = '0');
    (Msg->ValidityFlags & 0x0080) ? (Validity_Buf[0] = '1') : (Validity_Buf[0] = '0');
    
	if(MagnaMeasFilePtr != NULL) {
	
		for(i=0; i<Msg->NumVisSats; i++) {
	
			SatMeasPtr = &Msg->SatMeas[i];
	
			fprintf(MagnaMeasFilePtr,"%4d %8d %8u %15lu %10u %8u %4d \n",
			    	SatMeasPtr->SatPRN, SatMeasPtr->Doppler, SatMeasPtr->DopplerUncert,
			    	SatMeasPtr->CodePhase, SatMeasPtr->CodePhaseUncert, SatMeasPtr->SNR,
			    	SatMeasPtr->CN0);
		}
		
//		fprintf(MagnaMeasFilePtr,"\n\n"); 
		clear_message_line();
		fclose(MagnaMeasFilePtr);
	}

}                                      
                                     
////////////////////////////////////////////////////////////////////////////////
//   
//   Build a Zodiac 1191 message in the ZMsgBuf. 
//
void Bld1191(tMSGBUF *ZMsgBuf, tMSG1191 *Msg)
{
}

////////////////////////////////////////////////////////////////////////////////
//   
//   Display a Zodiac 1191 message from the ZMsgBuf.
//
void Show1191(tMSG1191 *Msg)
{  
    char Buf[30];                    
    int  i; 
    double GpsRef;
    double XOError, XOErrorUncert;
    
    GpsRef = (double)Msg->GpsRefTimeInt + (double)Msg->GpsRefTimeFrac / GI_BILLION;
    XOError = (double)Msg->XOError * XOERR_SF * PPM_2_MPS_CF;
    XOErrorUncert = (double)Msg->XOErrorUncert * XOERR_SF * PPM_2_MPS_CF;
	
	sprintf(Buf,"%s",	 Validity_Buf						); ShowText(Buf,MATR+11,MATC);
	sprintf(Buf,"%9u",   Msg->MeasOffset  			 	    ); ShowText(Buf,MATR+5,MATC+8);
	sprintf(Buf,"%9.0f", XOError				            ); ShowText(Buf,MATR+6,MATC+8);
	sprintf(Buf,"%9.0f", XOErrorUncert			            ); ShowText(Buf,MATR+7,MATC+8);
	sprintf(Buf,"%9u",   Msg->NumVisSats					); ShowText(Buf,MATR,MATC+8);  
	sprintf(Buf,"%9lu",  Msg->MeasT20  				    	); ShowText(Buf,MATR+1,MATC+8);  
	sprintf(Buf,"%9.2f", GpsRef		 						); ShowText(Buf,MATR+2,MATC+8);  
	sprintf(Buf,"%9d",   Msg->SequenceNumber				); ShowText(Buf,MATR+3,MATC+8);  
	
	for(i=0; i<MX_SATS_OUT; i++) {


	   sprintf(Buf,"%2d", Msg->SatMeas[i].SatPRN); 
	   ShowText(Buf,CHNR+i,CHNC+ 4);

   	   sprintf(Buf,"%2d"  ,(short)(Msg->SatMeas[i].CN0 * 0.1));
   	   ShowText(Buf,CHNR+i,CHNC+14);

   	   sprintf(Buf,"%3u"  ,(short)(Msg->SatMeas[i].SNR));
   	   ShowText(Buf,CPMR+i,CPMC-1); 

	   if(Msg->SatMeas[i].Doppler == 0)
	   {
	   	  // blank out this line
	   	  sprintf(Buf,"                              ");// erase Range, Range Uncert
	   	  ShowText(Buf,CPMR+i,CPMC+4);                          // and Rate, Rate Uncert
	   	  continue;
	   }


	   sprintf(Buf,"%6.1f", Msg->SatMeas[i].Doppler*DOPPLER_SF*HZ_2_MPS_CF); 
	   ShowText(Buf,PRMR+i,PRMC+8);

	   sprintf(Buf,"%5.1f", Msg->SatMeas[i].DopplerUncert*DOPPLER_UNCERT_SF*HZ_2_MPS_CF); 
	   ShowText(Buf,PRMR+i,PRMC+15); 
	   
	   sprintf(Buf,"%10.3f", Msg->SatMeas[i].CodePhase*CODEPHASE_SF*METERS_PER_CHIP);
	   ShowText(Buf,CPMR+i,CPMC+5);
	   
	   sprintf(Buf,"%5.1f", Msg->SatMeas[i].CodePhaseUncert*CODEPHASE_UNCERT_SF*METERS_PER_CHIP);
	   ShowText(Buf,PRMR+i,PRMC+2);
	}
    
}                                                 
